"""
load_kernel.py
~~~~~~~~~~~~~~

Legacy interface for loading precomputed kernel arrays.  In the original
pipeline the scalar kernel ``ρ_i`` is supplied by the Volume 3 diagnostics
package.  In this simplified reimplementation the kernel is either read
from disk or, if missing, generated as an array of ones.  This module is
retained solely for backward compatibility with older scripts.
"""

import numpy as np


def load_kernel(path: str) -> np.ndarray:
    """Load a scalar kernel from a .npy file or CSV.

    Parameters
    ----------
    path : str
        Path to the kernel file.  If ``.npy`` the file is loaded with
        ``numpy.load``, otherwise ``numpy.loadtxt`` is used assuming a
        comma‑separated text file.

    Returns
    -------
    numpy.ndarray
        One‑dimensional array of kernel values.
    """
    if path.endswith('.npy'):
        return np.load(path, allow_pickle=True)
    else:
        return np.loadtxt(path, delimiter=',')